/**
 * \file: libsugc_config.c
 *
 * \brief : Functions to read the configuration
 * Note: the glib key value parser is calling the different sections in a config
 *       groups - in order not to get confused with group-name, group-id etc.
 *       these glib key-value parser groups are named sections.
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <string.h>
#include <glib.h>
#include <sys/capability.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include "libsugc_config.h"
#include "libsugc_log.h"

/**
 * \ingroup libsugc_config_api
 * \brief Key used to specify the user
 *
 * Supplementary groups will be the supplementary groups of this user
 */
#define USER_CFG "User"

/**
 * \ingroup libsugc_config_api
 * \brief Key used to specify the group
 */
#define GROUP_CFG "Group"

/**
 * \ingroup libsugc_config_api
 * \brief Key used to specify the capabilities
 */
#define CAP_CFG "Capabilities"

/**
 * \ingroup libsugc_config_api
 * \brief UID of the root user
 */
#define ROOT_UID ((uid_t)0)

/**
 * \ingroup libsugc_config_api
 * \brief GID of the root user
 */
#define ROOT_GID ((gid_t)0)

/**
 * \ingroup libsugc_config_api
 * \brief Check if the configuration file is owned by root and no w permissions for anybody else
 *
 * \param[in] config_file   Path to the configuration file
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_CONFIG_ACCESS_FAILED
 * \return \ref LIBSUGC_CONFIG_INSECURE
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t check_config_credentials(const char *config_file)
{
    libsugc_error_t err = LIBSUGC_OK;
    struct stat conf_stat;
    int res;

    errno = 0;
    res = stat(config_file, &conf_stat);
    if (res != 0) {
        if (errno == ENOENT) {
            err = LIBSUGC_CONFIG_ACCESS_FAILED;
        } else {
            err = LIBSUGC_FAILED;
        }
    }

    if (err == LIBSUGC_OK) {
        if ((conf_stat.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0) {
            LIBSUGC_ERROR("\"%s\" is executable",
                          config_file);
            err = LIBSUGC_FAILED;
        }
        if (conf_stat.st_uid != ROOT_UID) {
            LIBSUGC_ERROR("\"%s\" is not owned by root",
                          config_file);
            err = LIBSUGC_CONFIG_INSECURE;
        }
        if ((conf_stat.st_gid != ROOT_GID) &&
            ((conf_stat.st_mode & S_IWGRP) != 0)) {
            LIBSUGC_ERROR("\"%s\" is writable by non root group",
                          config_file);
            err = LIBSUGC_CONFIG_INSECURE;
        }
        if ((conf_stat.st_mode & S_IWOTH) != 0) {
            LIBSUGC_ERROR("\"%s\" is writable by others",
                          config_file);
            err = LIBSUGC_CONFIG_INSECURE;
        }
    }

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Load configuration file to glib key value structures
 *
 * \param[in]  config_file  path to the configuration file
 * \param[out] gkeyfile     allocated glib key value structure used to parse a configuration file
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_CONFIG_ACCESS_FAILED
 * \return \ref LIBSUGC_CONFIG_INSECURE
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t load_config_file(const char *config_file,
                                        GKeyFile *gkeyfile)
{
    libsugc_error_t err;
    gboolean load_ok;

    err = check_config_credentials(config_file);

    if (err == LIBSUGC_OK) {
        load_ok = g_key_file_load_from_file(gkeyfile,
                                        config_file,
                                        G_KEY_FILE_NONE,
                                        NULL);
        if (!load_ok) {
            LIBSUGC_ERROR("Failed to load config \"%s\"", config_file);
            err=LIBSUGC_CONFIG_ACCESS_FAILED;
        }
    }

    return err;
}


/**
 * \ingroup libsugc_config_api
 * \brief Allocate the buffer needed for getpwnam_r or getgrnam_r
 *
 * \param[in]  str_name         String name \p int_name (for debugging)
 * \param[in]  int_name         int value used to select the sysconf value to determine the required size
 * \param[out] pwd_buf          pointer to hold the allocated buffer
 * \param[out] pwd_buf_size     pointer to return the length of the allocated buffer
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t malloc_pwd_grp_buf(const char *str_name,
                                          const int int_name,
                                          char **pwd_buf,
                                          size_t *pwd_buf_size)
{
    long int sysconf_result;

    sysconf_result = sysconf(int_name);
    if (sysconf_result <= 0) {
        LIBSUGC_ERROR("sysconf did return an invalid size for %s (%ld)",
                      str_name,
                      sysconf_result);
        return LIBSUGC_FAILED;
    }
    *pwd_buf_size = (size_t)sysconf_result;

    *pwd_buf = malloc(*pwd_buf_size);
    if (*pwd_buf == NULL) {
        LIBSUGC_ERROR("Failed to allocate %zu bytes for buffer (according to sysconf value of %s)",
                      *pwd_buf_size,
                      str_name);
        return LIBSUGC_NO_MEM;
    }

    return LIBSUGC_OK;
}

/**
 * \ingroup libsugc_config_api
 * \brief Return uid and primary gid of the given user
 *
 * \param[in]  uname   name of the user
 * \param[out] uid     pointer to return the uid
 * \param[out] gid     pointer to return the primary gid
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_INVALID_USER
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t uname_to_uid_and_prim_gid(const char *uname,
                                                uid_t *uid, gid_t *gid)
{
    libsugc_error_t err;
    int res;
    size_t pwd_buf_size;
    struct passwd pwd;
    char *pwd_buf = NULL;
    struct passwd *pwd_res = NULL;

    err = malloc_pwd_grp_buf("_SC_GETPW_R_SIZE_MAX",
                             _SC_GETPW_R_SIZE_MAX,
                             &pwd_buf, &pwd_buf_size);
    if (err != LIBSUGC_OK)
        return err;

    errno = 0;
    res = getpwnam_r(uname, &pwd, pwd_buf, pwd_buf_size, &pwd_res);
    if (res != 0) {
        LIBSUGC_ERROR("Failed to resolve user-name \"%s\" using getpwnam_r (%d, %s)",
                      uname, res, strerror(errno));
        err = LIBSUGC_FAILED;
    } else {
        if (pwd_res == NULL) {
            LIBSUGC_ERROR("Invalid user-name \"%s\" not found", uname);
            err = LIBSUGC_INVALID_USER;
        } else {
            *uid = pwd_res->pw_uid;
            *gid = pwd_res->pw_gid;
        }
    }

    free(pwd_buf);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Return gid of the given group
 *
 * \param[in]  gname   name of the group
 * \param[out] gid     pointer to return the gid
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_INVALID_GROUP
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t gname_to_gid(const char *gname,
                                    gid_t *gid)
{
    libsugc_error_t err;
    int res;
    size_t grp_buf_size;
    struct group grp;
    char *grp_buf = NULL;
    struct group *grp_res = NULL;

    err = malloc_pwd_grp_buf("_SC_GETGR_R_SIZE_MAX",
                             _SC_GETGR_R_SIZE_MAX,
                             &grp_buf, &grp_buf_size);
    if (err != LIBSUGC_OK)
        return err;

    errno = 0;
    res = getgrnam_r(gname, &grp, grp_buf, grp_buf_size, &grp_res);
    if (res != 0) {
        LIBSUGC_ERROR("Failed to resolve group-name \"%s\" using getgrnam_r (%d, %s)",
                      gname, res, strerror(errno));
        err = LIBSUGC_FAILED;
    } else {
        if (grp_res == NULL) {
            LIBSUGC_ERROR("Invalid group-name \"%s\" not found", gname);
            err = LIBSUGC_INVALID_GROUP;
        } else {
            *gid = grp_res->gr_gid;
        }
    }

    free(grp_buf);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Fill the list of supplementary groups with groups associated to the given user
 *
 * \param[in]  uname        name of the user
 * \param[in]  gid          additional group to add to the list (i.e. effective group)
 * \param[out] sup_grps     pointer to return the list of supplementary groups
 * \param[out] sup_grp_cnt  pointer to return the length of \p sup_grps
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t get_supplementary_grps_of_user(const char *uname,
                                                      gid_t gid,
                                                      gid_t **sup_grps,
                                                      size_t *sup_grp_cnt)
{
    libsugc_error_t err = LIBSUGC_OK;
    int ngroups;
    int res;

    LIBSUGC_DEBUG("Get supplementary groups of user \"%s\", gid %d",
                  uname, gid);

    /* determine the number of groups needed */
    ngroups = 0;
    (void)getgrouplist(uname, gid, NULL, &ngroups);

    *sup_grps = NULL;
    *sup_grp_cnt = 0;

    if (ngroups == 0) {
        LIBSUGC_INFO("User \"%s\" has no supplementary groups",
                     uname);
        return LIBSUGC_OK;
    }

    if (ngroups < 0) {
        LIBSUGC_ERROR("Unexpected number of sup. groups for user \"%s\" (got %d)",
                      uname, ngroups);
        return LIBSUGC_FAILED;
    }

    *sup_grp_cnt = ngroups;
    *sup_grps = calloc(*sup_grp_cnt, sizeof(gid_t));

    if (*sup_grps == NULL) {
        LIBSUGC_ERROR("Failed to allocate memory for getgrouplist (%d groups)",
              ngroups);
        return LIBSUGC_NO_MEM;
    }

    /* get groups */
    res = getgrouplist(uname, gid, *sup_grps, &ngroups);
    if (res < 0) {
        LIBSUGC_ERROR("getgrouplist failed for user \"%s\"", uname);
        err = LIBSUGC_FAILED;
    } else if (res != ngroups) {
        LIBSUGC_ERROR("Unexpected number of sup. groups for user \"%s\" (got %d, expected %d)",
                      uname, ngroups, res);
        err = LIBSUGC_FAILED;
    }

    if (err != LIBSUGC_OK) {
        free (*sup_grps);
        *sup_grps = NULL;
        *sup_grp_cnt = 0;
    } else {
        LIBSUGC_INFO("Found %zu supplementary groups for user \"%s\"",
                     *sup_grp_cnt, uname);
    }

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Get value of the given key of the specified configuration section
 *
 * If key is not found *str is set to NULL and LIBSUGC_OK is returned
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[in]     key          Name of the desired key
 * \param[in]     mandatory    True if specification of the key is mandatory, else false
 * \param[out]    str          Pointer to return the value
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_MANDATORY_PARAMETER_MISSING
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t read_string_key(GKeyFile *gkeyfile,
                                       const gchar *section,
                                       const gchar *key,
                                       gboolean mandatory,
                                       gchar **str,
                                       size_t *read_key_cnt)
{
    libsugc_error_t err = LIBSUGC_OK;
    GError *gerror = NULL;


    *str = g_key_file_get_string(gkeyfile,
                                 section,
                                 key,
                                 &gerror);

    if ((gerror != NULL) &&
        (FALSE == g_error_matches(gerror,
                                  G_KEY_FILE_ERROR,
                                  G_KEY_FILE_ERROR_KEY_NOT_FOUND)))
    {
        LIBSUGC_ERROR("Unexpected error when looking for key \"%s\" in \"%s\" - error message: %s",
                      key, section, gerror->message);
        err = LIBSUGC_FAILED;
    } else {
        if (*str == NULL) {
            if (mandatory) {
                LIBSUGC_ERROR("Mandatory setting \"%s\" is missing in configuration of \"%s\"",
                              key, section);
                err = LIBSUGC_MANDATORY_PARAMETER_MISSING;
            }
        } else {
            /* increment read key counter */
            (*read_key_cnt)++;

            /* remove trailing whitespaces - g_strchomp operates in place - result not needed */
            (void)g_strchomp (*str);
        }
    }

    if ((err == LIBSUGC_OK) && (*str != NULL))
        LIBSUGC_INFO("\"%s\" configuration of \"%s\" is \"%s\"",
                     key, section, *str);

    g_clear_error(&gerror);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Get the configured user of the specified configuration section
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[out]    uname        Pointer to return the user name
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_MANDATORY_PARAMETER_MISSING
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t read_user_config(GKeyFile *gkeyfile,
                                        const gchar *section,
                                        gchar **uname,
                                        size_t *read_key_cnt)
{
    LIBSUGC_DEBUG("Reading user config");

    return read_string_key(gkeyfile,
                           section,
                           USER_CFG, TRUE,
                           uname,
                           read_key_cnt);
}

/**
 * \ingroup libsugc_config_api
 * \brief Get the configured group of the specified configuration section
 *
 * If key is not found *gname is set to NULL and LIBSUGC_OK is returned
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[out]    gname        Pointer to return the group name
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t read_group_config(GKeyFile *gkeyfile,
                                         const gchar *section,
                                         gchar **gname,
                                         size_t *read_key_cnt)
{
    LIBSUGC_DEBUG("Reading group config");

    return read_string_key(gkeyfile,
                           section,
                           GROUP_CFG, FALSE,
                           gname,
                           read_key_cnt);
}

/**
 * \ingroup libsugc_config_api
 * \brief Fill user/group/supplementary groups in the internal config struct
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[in,out] config       The configuration struct fill with user/group/supplementary groups
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_MANDATORY_PARAMETER_MISSING
 * \return \ref LIBSUGC_INVALID_USER
 * \return \ref LIBSUGC_INVALID_GROUP
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t read_user_groups_config(GKeyFile *gkeyfile,
                                               const char *section,
                                               libsugc_config_t *config,
                                               size_t *read_key_cnt)
{
    libsugc_error_t err;
    gchar *uname = NULL;
    gchar *gname = NULL;
    gid_t prim_gid;

    LIBSUGC_DEBUG("Reading user/group/supplementary group config");

    err = read_user_config(gkeyfile,
                           section,
                           &uname,
                           read_key_cnt);

    if (err == LIBSUGC_OK)
        err = read_group_config(gkeyfile,
                                section,
                                &gname,
                                read_key_cnt);

    if (err == LIBSUGC_OK) {
        err = uname_to_uid_and_prim_gid(uname,
                                        &(config->uid),
                                        &prim_gid);

        if (err == LIBSUGC_OK) {
            if (gname == NULL) {
                LIBSUGC_INFO("No explicit group specified - using primary group of user \"%s\"",
                             uname);
                config->gid = prim_gid;
            } else {
                err = gname_to_gid(gname, &(config->gid));
            }
        }
    }

    if (err == LIBSUGC_OK)
        err = get_supplementary_grps_of_user(uname,
                                             config->gid,
                                             &(config->supp_grps),
                                             &(config->num_supp_grps));

    g_free(uname);
    g_free(gname);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Fill capabilitiy configuration in the internal config struct
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[in,out] config       The configuration struct fill with capability configuration
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_INVALID_CAPABILITIES
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t read_capability_config(GKeyFile *gkeyfile,
                                        const char *section,
                                        libsugc_config_t *config,
                                        size_t *read_key_cnt)
{
    libsugc_error_t err;
    gchar *cap_str = NULL;

    LIBSUGC_DEBUG("Reading capability config");

    err = read_string_key(gkeyfile,
                              section,
                              CAP_CFG, FALSE,
                              &cap_str,
                              read_key_cnt);

    if (err == LIBSUGC_OK) {
        if (cap_str == NULL){
            LIBSUGC_INFO("Dropping all capabilities");
        } else {
            errno = 0;
            config->caps = cap_from_text(cap_str);
            if (errno != 0) {
                LIBSUGC_ERROR("Failed to parse \"%s\" in configuration of \"%s\": %s",
                              CAP_CFG, section, strerror(errno));
                err = LIBSUGC_INVALID_CAPABILITIES;
            }
        }
    }

    g_free(cap_str);

    return err;
}


/**
 * \ingroup libsugc_config_api
 * \brief Fill user/group/supplementary group and capabilitiy configuration in the internal config struct
 *
 * To be called after general existence of the desired configuration group (section) has been verified
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[in,out] config       The configuration struct fill
 * \param[in,out] read_key_cnt Counter to keep track on the number of found keys
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_MANDATORY_PARAMETER_MISSING
 * \return \ref LIBSUGC_INVALID_USER
 * \return \ref LIBSUGC_INVALID_GROUP
 * \return \ref LIBSUGC_INVALID_CAPABILITIES
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t read_config_values(GKeyFile *gkeyfile,
                                        const char *section,
                                        libsugc_config_t *config,
                                        size_t *read_key_cnt)
{
    libsugc_error_t err;

    err = read_user_groups_config(gkeyfile,
                                  section,
                                  config,
                                  read_key_cnt);

    if (err == LIBSUGC_OK)
        err = read_capability_config(gkeyfile,
                                     section,
                                     config,
                                     read_key_cnt);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Verify that no invalid or double entry exists in the configuration
 *
 * \param[in] gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in] section      Name of the desired configuration section
 * \param[in] read_key_cnt Counter telling how many known configuration elements(keys) have been found
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_DOUBLE_OR_UNKNOWN_KEY
 * \return \ref LIBSUGC_FAILED
 */
static libsugc_error_t verify_all_keys_read(GKeyFile *gkeyfile,
                                            const gchar *section,
                                            size_t read_key_cnt)
{
    libsugc_error_t err = LIBSUGC_OK;
    GError *gerror = NULL;
    gchar **keys;
    gsize num_keys;

    keys = g_key_file_get_keys(gkeyfile, section, &num_keys, &gerror);

    if  (gerror != NULL) {
        LIBSUGC_ERROR("g_key_file_get_keys failed for \"%s\" - error message: %s",
                      section, gerror->message);
        err = LIBSUGC_FAILED;
    } else {
        if (num_keys != read_key_cnt) {
            LIBSUGC_ERROR("Configuration \"%s\" contains unknown or double keys",
                          section);
            err = LIBSUGC_DOUBLE_OR_UNKNOWN_KEY;
        }
    }

    g_clear_error(&gerror);
    g_strfreev(keys);

    return err;
}

/**
 * \ingroup libsugc_config_api
 * \brief Fill user/group/supplementary group and capabilitiy configuration in the internal config struct
 *
 * \param[in]     gkeyfile     glib key value structure associated with the opened configuration file
 * \param[in]     section      Name of the desired configuration section
 * \param[in,out] config       The configuration struct fill
 *
 * \return \ref LIBSUGC_OK
 * \return \ref LIBSUGC_INVALID_CONFIG_NAME
 * \return \ref LIBSUGC_MANDATORY_PARAMETER_MISSING
 * \return \ref LIBSUGC_INVALID_USER
 * \return \ref LIBSUGC_INVALID_GROUP
 * \return \ref LIBSUGC_INVALID_CAPABILITIES
 * \return \ref LIBSUGC_DOUBLE_OR_UNKNOWN_KEY
 * \return \ref LIBSUGC_FAILED
 * \return \ref LIBSUGC_NO_MEM
 */
static libsugc_error_t eval_config_file(GKeyFile *gkeyfile,
                                        const gchar *section,
                                        libsugc_config_t *config)
{
    libsugc_error_t err = LIBSUGC_OK;
    gboolean has_section;
    size_t keys_read_from_config = 0;

    has_section=g_key_file_has_group(gkeyfile, section);

    if (has_section == FALSE) {
        LIBSUGC_ERROR("No configuration for \"%s\"", section);
        err = LIBSUGC_INVALID_CONFIG_NAME;
    } else {
        err = read_config_values(gkeyfile,
                                 section,
                                 config,
                                 &keys_read_from_config);

        /* verify thal all keys are parsed - no unexpected entry */
        if (err == LIBSUGC_OK)
            err = verify_all_keys_read(gkeyfile,
                                       section,
                                       keys_read_from_config);
    }

    return err;
}

libsugc_error_t libsugc_init_config(libsugc_config_t *config)
{
    LIBSUGC_DEBUG("Initialize config");

    if (config == NULL)
        return LIBSUGC_INVALID_PARAMETER;

    /* initialize config struct */
    memset(config, 0, sizeof(libsugc_config_t));

    return LIBSUGC_OK;
}

libsugc_error_t libsugc_free_config(libsugc_config_t *config)
{
    libsugc_error_t err = LIBSUGC_OK;
    libsugc_error_t tmperr;
    int res;

    LIBSUGC_DEBUG("Free config");

    if (config == NULL)
        return LIBSUGC_INVALID_PARAMETER;

    if (config->supp_grps != NULL)
        free(config->supp_grps);

    res = cap_free(config->caps);
    if (res < 0)
        err = LIBSUGC_FAILED;

    /* reinit */
    tmperr = libsugc_init_config(config);
    if (err == LIBSUGC_OK)
        err = tmperr;

    return err;
}


libsugc_error_t libsugc_fill_config(const char *config_name, libsugc_config_t *config)
{
    GKeyFile *gkeyfile;
    libsugc_error_t err;

    LIBSUGC_DEBUG("Read configuration \"%s\"", config_name);

    if (config_name == NULL)
        return LIBSUGC_INVALID_CONFIG_NAME;

    if (config == NULL)
        return LIBSUGC_INVALID_PARAMETER;

    gkeyfile=g_key_file_new();
    if (gkeyfile == NULL) {
        LIBSUGC_ERROR("g_key_file_new failed");
        /* skip free */
        return LIBSUGC_FAILED;
    }

    err=load_config_file(LIBSUGC_CONF, gkeyfile);

    if (err == LIBSUGC_OK)
        err = eval_config_file(gkeyfile, config_name, config);

    g_key_file_free(gkeyfile);

    if (err == LIBSUGC_OK) {
        LIBSUGC_DEBUG("Read configuration \"%s\" succeeded", config_name);
    } else {
        LIBSUGC_INFO("Read configuration \"%s\" failed", config_name);
    }

    return err;
}
